import argparse
from continual_rl.utils.configuration_loader import ConfigurationLoader
from continual_rl.available_policies import get_available_policies
from continual_rl.experiment_specs import get_available_experiments


class ArgumentMissingException(Exception):
    def __init__(self, error_str):
        super().__init__(error_str)


class ArgparseManager(object):
    """
    Handles processing the command line, and then calls the ConfigurationLoader to actually load the experiment
    and policy as appropriate.
    """

    def __init__(self):
        self.command_line_mode_parser = self._create_command_line_mode_parser()
        self.config_mode_parser = self._create_config_mode_parser()

    @classmethod
    def _create_command_line_mode_parser(cls):
        # All other arguments will be converted to a dictionary and used the same way as if it were a configuration
        command_line_parser = argparse.ArgumentParser()
        command_line_parser.add_argument("--output-dir", help="The output directory where this experiment's results"
                                                              "(logs and models) will be stored.",
                                         type=str, default="tmp")

        return command_line_parser

    @classmethod
    def _create_config_mode_parser(cls):
        """
        If the "config-file" mode is run, these are the arguments expected.
        Example: python main.py --config-file path/to/my_config.json --output-dir path/to/output
        """
        config_parser = argparse.ArgumentParser()
        config_parser.add_argument('--config-file', type=str, help='The full path to the JSON file containing the '
                                                                   'experiment configs.')
        config_parser.add_argument("--output-dir", help="The output directory where logs and models for all experiments"
                                                        " generated by this config file are stored.",
                                   type=str, default="tmp")
        config_parser.add_argument("--resume-id", help="The id of the experiment to resume",
                                   type=int, default=None)
        return config_parser

    @classmethod
    def parse(cls, raw_args):
        # Load the available policies and experiments
        available_policies = get_available_policies()
        available_experiments = get_available_experiments()

        argparser = ArgparseManager()
        configuration_loader = ConfigurationLoader(available_policies=available_policies,
                                                   available_experiments=available_experiments)

        args, extras = argparser.config_mode_parser.parse_known_args(raw_args)

        # If we successfully parse a config_file, enter config-mode
        if args.config_file is not None:
            assert len(extras) == 0, f"Unknown arguments found: {extras}"
            print(f"Entering config mode using file {args.config_file} and output directory {args.output_dir}")

            if args.resume_id is not None:
                print(f"Resuming from experiment id {args.resume_id}")

            experiment, policy = configuration_loader.load_next_experiment_from_config(args.output_dir,
                                                                                       args.config_file,
                                                                                       resume_id=args.resume_id)
        else:
            # otherwise default to command-line mode and use command line parser
            args, extras = argparser.command_line_mode_parser.parse_known_args(raw_args)

            # Extras is a list in the form ["--arg1", "val1", "--arg2", "val2"]. Convert it to a dictionary
            raw_experiment = {extras[i].replace('--', ''): extras[i + 1] for i in range(0, len(extras), 2)}

            if "experiment" not in raw_experiment:
                raise ArgumentMissingException("--experiment required in command-line mode")

            if "policy" not in raw_experiment:
                raise ArgumentMissingException("--policy required in command-line mode")

            # load_next_experiment is expecting a list of experiment configs, so put our experiment in a list
            experiment, policy = configuration_loader.load_next_experiment_from_dicts(args.output_dir,
                                                                                     [raw_experiment])


        # Example usage:
        # policy_id = "ppo"  # Select a policy by its ID
        # experiment_id = "minihack_nav_paired_2_cycles"  # Select an experiment by its ID
        # # Check if the specified policy exists
        # if policy_id not in available_policies:
        #     print(f"Policy {policy_id} not found in available policies.")
        #
        # # Check if the specified experiment exists
        # if experiment_id not in available_experiments:
        #     print(f"Experiment {experiment_id} not found in available experiments.")
        #
        # # Load the selected policy and experiment
        # policy_class = available_policies[policy_id].policy
        # policy_config_class = available_policies[policy_id].config
        # policy_config = policy_config_class()  # Use default configuration for the policy
        #
        # experiment = available_experiments[experiment_id]  # Get the experiment by ID
        #
        # # Initialize the policy with its config, observation space, and action spaces from the experiment
        # policy = policy_class(policy_config, experiment.observation_space, experiment.action_spaces)
        #
        # # Set the task IDs for the policy if needed
        # policy.set_task_ids(experiment.task_ids)

        return experiment, policy



class ArgparseManager2(object):
    """
    Handles processing the parameters within the code, and then calls the ConfigurationLoader to actually load
    the experiment and policy as appropriate.
    """

    def __init__(self):
        # You can initialize any needed defaults here
        self.output_dir = "My_sane_atari(frozen)/tmp"
        self.config_file = None
        self.resume_id = None

    def choose_parameters(self):
        """
        Instead of parsing from command line, this function allows you to set parameters directly in code.
        """
        # Example: directly setting the configuration for experiment and policy
        self.output_dir = "/tmp"  # set the output directory
        self.config_file = None  # set the path to the config file
        self.resume_id = 1  # set an experiment ID if you want to resume

        # Additional manual parameters for command-line mode
        raw_experiment = {
            # "experiment": "atari_6_tasks_5_cycles",  # specify the experiment name
            "experiment": "minihack_nav_paired_2_cycles",
            "policy": "DSNet",  # specify the policy name
            # Add more parameters as needed
        }

        return raw_experiment

    def run(self):
        """
        This function handles the logic of running the experiment without command-line parsing.
        """
        # Load the available policies and experiments
        available_policies = get_available_policies()
        available_experiments = get_available_experiments()

        # Instantiate configuration loader
        configuration_loader = ConfigurationLoader(available_policies=available_policies,
                                                   available_experiments=available_experiments)

        # If using a config file
        if self.config_file is not None:
            print(f"Entering config mode using file {self.config_file} and output directory {self.output_dir}")

            if self.resume_id is not None:
                print(f"Resuming from experiment id {self.resume_id}")

            experiment, policy = configuration_loader.load_next_experiment_from_config(
                self.output_dir, self.config_file, resume_id=self.resume_id
            )
        else:
            # Otherwise, run with manually selected command-line parameters
            raw_experiment = self.choose_parameters()

            r_experiment = raw_experiment['experiment']

            if "experiment" not in raw_experiment:
                raise ArgumentMissingException("--experiment required in manual mode")

            if "policy" not in raw_experiment:
                raise ArgumentMissingException("--policy required in manual mode")

            # Load the experiment based on the parameters set in `choose_parameters`
            experiment, policy = configuration_loader.load_next_experiment_from_dicts(
                self.output_dir, [raw_experiment]
            )

        # # Now that the experiment and policy are loaded, you can proceed with running them
        # experiment.try_run(policy)

        return experiment, policy, r_experiment





